const { supabase, STORAGE_BUCKET } = require('../config/database');
const { v4: uuidv4 } = require('uuid');
const path = require('path');
const AdmZip = require('adm-zip');
const logService = require('./logService');
const discordService = require('./discordService');

class FileService {
    /**
     * Upload a file to Supabase Storage
     */
    async uploadFile(file, filePath, userId, username) {
        try {
            // Generate unique storage path
            const fileExt = path.extname(file.originalname);
            const fileName = file.originalname;
            const storagePath = `${filePath}/${uuidv4()}${fileExt}`;

            // Upload to Supabase Storage
            const { data: uploadData, error: uploadError } = await supabase.storage
                .from(STORAGE_BUCKET)
                .upload(storagePath, file.buffer, {
                    contentType: file.mimetype,
                    upsert: false
                });

            if (uploadError) {
                throw uploadError;
            }

            // Create file record in database
            const { data: fileRecord, error: dbError } = await supabase
                .from('files')
                .insert({
                    name: fileName,
                    path: filePath,
                    type: 'file',
                    size: file.size,
                    owner_id: userId,
                    storage_path: storagePath,
                    is_deleted: false
                })
                .select()
                .single();

            if (dbError) {
                // Rollback storage upload
                await supabase.storage.from(STORAGE_BUCKET).remove([storagePath]);
                throw dbError;
            }

            // Log activity
            await logService.createLog(userId, 'upload', fileName, { path: filePath, size: file.size });
            await discordService.logFileUpload(username, fileName, file.size, filePath);

            return fileRecord;
        } catch (error) {
            console.error('Error uploading file:', error);
            throw error;
        }
    }

    /**
     * Download a file from Supabase Storage
     */
    async downloadFile(fileId) {
        try {
            // Get file record
            const { data: fileRecord, error: dbError } = await supabase
                .from('files')
                .select('*')
                .eq('id', fileId)
                .single();

            if (dbError || !fileRecord) {
                throw new Error('File not found');
            }

            if (fileRecord.type === 'folder') {
                throw new Error('Cannot download a folder');
            }

            // Download from storage
            const { data: fileData, error: storageError } = await supabase.storage
                .from(STORAGE_BUCKET)
                .download(fileRecord.storage_path);

            if (storageError) {
                throw storageError;
            }

            return {
                data: fileData,
                name: fileRecord.name,
                size: fileRecord.size
            };
        } catch (error) {
            console.error('Error downloading file:', error);
            throw error;
        }
    }

    /**
     * List files and folders in a path (excluding deleted)
     */
    async listFiles(filePath = '/', userId = null) {
        try {
            let query = supabase
                .from('files')
                .select(`
          *,
          users:owner_id (username)
        `)
                .eq('path', filePath)
                .eq('is_deleted', false)
                .order('type', { ascending: false }) // Folders first
                .order('name', { ascending: true });

            const { data, error } = await query;

            if (error) {
                throw error;
            }

            return data;
        } catch (error) {
            console.error('Error listing files:', error);
            throw error;
        }
    }

    /**
     * List deleted files (Trash)
     */
    async listTrash(userId) {
        try {
            let query = supabase
                .from('files')
                .select(`
          *,
          users:owner_id (username)
        `)
                .eq('is_deleted', true)
                .order('deleted_at', { ascending: false });

            // Optionally filter by owner if not admin? 
            // For now assuming all users see global trash or user specific?
            // "My Files" implies user specific. Let's filter by owner_id if provided.
            if (userId) {
                // query = query.eq('owner_id', userId); 
                // We keep it global for now based on current logic, but maybe specific to user?
                // The current app seems to share files.
            }

            const { data, error } = await query;

            if (error) {
                throw error;
            }

            return data;
        } catch (error) {
            console.error('Error listing trash:', error);
            throw error;
        }
    }

    /**
     * Create a folder
     */
    async createFolder(folderName, folderPath, userId, username) {
        try {
            const { data, error } = await supabase
                .from('files')
                .insert({
                    name: folderName,
                    path: folderPath,
                    type: 'folder',
                    size: 0,
                    owner_id: userId,
                    storage_path: null,
                    is_deleted: false
                })
                .select()
                .single();

            if (error) {
                throw error;
            }

            // Log activity
            await logService.createLog(userId, 'create_folder', folderName, { path: folderPath });
            await discordService.logFolderCreate(username, folderName, folderPath);

            return data;
        } catch (error) {
            console.error('Error creating folder:', error);
            throw error;
        }
    }

    /**
     * Rename a file or folder
     */
    async renameFile(fileId, newName, userId, username) {
        try {
            // Get current file
            const { data: currentFile, error: fetchError } = await supabase
                .from('files')
                .select('*')
                .eq('id', fileId)
                .single();

            if (fetchError || !currentFile) {
                throw new Error('File not found');
            }

            const oldName = currentFile.name;

            // Update file name
            const { data, error } = await supabase
                .from('files')
                .update({ name: newName })
                .eq('id', fileId)
                .select()
                .single();

            if (error) {
                throw error;
            }

            // Log activity
            await logService.createLog(userId, 'rename', newName, { oldName, path: currentFile.path });
            await discordService.logFileRename(username, oldName, newName, currentFile.path);

            return data;
        } catch (error) {
            console.error('Error renaming file:', error);
            throw error;
        }
    }

    /**
     * Move a file or folder
     */
    async moveFile(fileId, newPath, userId, username) {
        try {
            // Get current file
            const { data: currentFile, error: fetchError } = await supabase
                .from('files')
                .select('*')
                .eq('id', fileId)
                .single();

            if (fetchError || !currentFile) {
                throw new Error('File not found');
            }

            const oldPath = currentFile.path;

            // Update file path
            const { data, error } = await supabase
                .from('files')
                .update({ path: newPath })
                .eq('id', fileId)
                .select()
                .single();

            if (error) {
                throw error;
            }

            // Log activity
            await logService.createLog(userId, 'move', currentFile.name, { from: oldPath, to: newPath });
            await discordService.logFileMove(username, currentFile.name, oldPath, newPath);

            return data;
        } catch (error) {
            console.error('Error moving file:', error);
            throw error;
        }
    }

    /**
     * Soft Delete a file or folder (Move to Trash)
     */
    async deleteFile(fileId, userId, username) {
        try {
            const { data: fileRecord, error: fetchError } = await supabase
                .from('files')
                .select('*')
                .eq('id', fileId)
                .single();

            if (fetchError || !fileRecord) {
                throw new Error('File not found');
            }

            // Soft delete: set is_deleted = true
            const { error: updateError } = await supabase
                .from('files')
                .update({
                    is_deleted: true,
                    deleted_at: new Date()
                })
                .eq('id', fileId);

            if (updateError) {
                throw updateError;
            }

            // Log activity
            await logService.createLog(userId, 'delete', fileRecord.name, { path: fileRecord.path });
            await discordService.logFileDelete(username, fileRecord.name, fileRecord.path);

            return true;
        } catch (error) {
            console.error('Error deleting file:', error);
            throw error;
        }
    }

    /**
     * Restore file from Trash
     */
    async restoreFile(fileId, userId, username) {
        try {
            const { error: updateError } = await supabase
                .from('files')
                .update({
                    is_deleted: false,
                    deleted_at: null
                })
                .eq('id', fileId);

            if (updateError) {
                throw updateError;
            }

            await logService.createLog(userId, 'restore', fileId, {});
            return true;
        } catch (error) {
            console.error('Error restoring file:', error);
            throw error;
        }
    }

    /**
     * Permanently Delete (Empty Trash or specific file)
     */
    async permanentDelete(fileId, userId, username) {
        try {
            // Get file record
            const { data: fileRecord, error: fetchError } = await supabase
                .from('files')
                .select('*')
                .eq('id', fileId)
                .single();

            if (fetchError || !fileRecord) throw new Error('File not found');

            // If it's a file, delete from storage
            if (fileRecord.type === 'file' && fileRecord.storage_path) {
                const { error: storageError } = await supabase.storage
                    .from(STORAGE_BUCKET)
                    .remove([fileRecord.storage_path]);

                if (storageError) console.error('Error deleting from storage:', storageError);
            }

            // Delete from database
            const { error: dbError } = await supabase
                .from('files')
                .delete()
                .eq('id', fileId);

            if (dbError) throw dbError;

            return true;
        } catch (error) {
            console.error('Error in permanent delete:', error);
            throw error;
        }
    }

    /**
     * Search files by name
     */
    async searchFiles(searchTerm, userId = null) {
        try {
            let query = supabase
                .from('files')
                .select(`
          *,
          users:owner_id (username)
        `)
                .ilike('name', `%${searchTerm}%`)
                .eq('is_deleted', false)
                .order('name', { ascending: true })
                .limit(50);

            const { data, error } = await query;

            if (error) {
                throw error;
            }

            return data;
        } catch (error) {
            console.error('Error searching files:', error);
            throw error;
        }
    }

    /**
     * Get file by ID
     */
    async getFileById(fileId) {
        try {
            const { data, error } = await supabase
                .from('files')
                .select(`
          *,
          users:owner_id (username)
        `)
                .eq('id', fileId)
                .single();

            if (error) {
                throw error;
            }

            return data;
        } catch (error) {
            console.error('Error getting file:', error);
            throw error;
        }
    }

    /**
     * Get storage statistics
     */
    async getStorageStats(userId = null) {
        try {
            let query = supabase
                .from('files')
                .select('type, size')
                .eq('is_deleted', false);

            if (userId) {
                query = query.eq('owner_id', userId);
            }

            const { data, error } = await query;

            if (error) {
                throw error;
            }

            const stats = {
                totalFiles: 0,
                totalFolders: 0,
                totalSize: 0
            };

            data.forEach(file => {
                if (file.type === 'file') {
                    stats.totalFiles++;
                    stats.totalSize += file.size || 0;
                } else {
                    stats.totalFolders++;
                }
            });

            return stats;
        } catch (error) {
            console.error('Error getting storage stats:', error);
            return { totalFiles: 0, totalFolders: 0, totalSize: 0 };
        }
    }

    /**
     * Copy a file
     */
    async copyFile(fileId, destPath, userId, username) {
        try {
            // Get original file
            const { data: originalFile, error: fetchError } = await supabase
                .from('files')
                .select('*')
                .eq('id', fileId)
                .single();

            if (fetchError || !originalFile) {
                throw new Error('File not found');
            }

            if (originalFile.type === 'folder') {
                throw new Error('Copying folders is not supported yet');
            }

            // Generate new storage path (copy)
            const fileExt = path.extname(originalFile.name);
            const newStoragePath = `${destPath}/${uuidv4()}${fileExt}`;

            // Copy in storage
            const { data: copyData, error: copyError } = await supabase.storage
                .from(STORAGE_BUCKET)
                .copy(originalFile.storage_path, newStoragePath);

            if (copyError) {
                throw copyError;
            }

            // Insert new DB record
            const { data, error } = await supabase
                .from('files')
                .insert({
                    name: `Copy of ${originalFile.name}`,
                    path: destPath,
                    type: 'file',
                    size: originalFile.size,
                    owner_id: userId,
                    storage_path: newStoragePath,
                    is_deleted: false
                })
                .select()
                .single();

            if (error) {
                await supabase.storage.from(STORAGE_BUCKET).remove([newStoragePath]);
                throw error;
            }

            // Log activity
            await logService.createLog(userId, 'copy', originalFile.name, { from: originalFile.path, to: destPath });
            await discordService.logFileMove(username, originalFile.name, originalFile.path, destPath + ' (Copy)');

            return data;
        } catch (error) {
            console.error('Error copying file:', error);
            throw error;
        }
    }

    /**
     * Compress files
     */
    async compressFiles(fileIds, archiveName, currentPath, userId, username) {
        try {
            const zip = new AdmZip();

            // 1. Fetch selected items (files and folders)
            console.log(`Compressing ${fileIds.length} items...`);
            const { data: initialItems, error } = await supabase
                .from('files')
                .select('*')
                .in('id', fileIds);

            if (error) {
                console.error('Database error fetching items:', error);
                throw new Error('Database error fetching items: ' + error.message);
            }
            if (!initialItems.length) {
                console.warn('No items found for compression');
                throw new Error('No items selected for compression');
            }

            const filesToZip = [];

            // 2. Process items (expand folders)
            for (const item of initialItems) {
                if (item.type === 'file') {
                    filesToZip.push({
                        ...item,
                        zipPath: item.name
                    });
                } else if (item.type === 'folder') {
                    console.log(`Processing folder: ${item.name}`);
                    const folderPath = item.path === '/' ? `/${item.name}` : `${item.path}/${item.name}`;

                    // Fetch descendants
                    const { data: descendants, error: descError } = await supabase
                        .from('files')
                        .select('*')
                        .eq('type', 'file')
                        .eq('is_deleted', false)
                        .or(`path.eq.${folderPath},path.like.${folderPath}/%`);

                    if (descError) {
                        console.error(`Error fetching descendants for ${item.name}:`, descError);
                        continue;
                    }

                    if (descendants) {
                        console.log(`Found ${descendants.length} files in ${item.name}`);
                        descendants.forEach(f => {
                            // Calculate relative path
                            let relativeDir = f.path.substring(folderPath.length);
                            if (relativeDir.startsWith('/')) relativeDir = relativeDir.substring(1);

                            // Construct zip path: FolderName/Relative/FileName
                            const zipParts = [item.name];
                            if (relativeDir) zipParts.push(relativeDir);
                            zipParts.push(f.name);
                            const zipPath = zipParts.join('/');

                            filesToZip.push({
                                ...f,
                                zipPath: zipPath
                            });
                        });
                    }
                }
            }

            if (!filesToZip.length) {
                throw new Error('No files found to compress');
            }

            console.log(`Total files to zip: ${filesToZip.length}`);

            // 3. Download files and add to zip
            for (const file of filesToZip) {
                console.log(`Downloading ${file.name} -> ${file.zipPath}...`);
                const { data: fileData, error: downError } = await supabase.storage
                    .from(STORAGE_BUCKET)
                    .download(file.storage_path);

                if (downError) {
                    console.error(`Error downloading ${file.name}:`, downError);
                    continue;
                }

                if (fileData) {
                    const buffer = Buffer.from(await fileData.arrayBuffer());
                    zip.addFile(file.zipPath, buffer);
                }
            }

            console.log('Finalizing zip...');
            const zipBuffer = zip.toBuffer();
            const fileName = archiveName.endsWith('.zip') ? archiveName : `${archiveName}.zip`;

            // Upload zip
            console.log(`Uploading zip ${fileName}, size: ${zipBuffer.length}...`);
            const uploadResult = await this.uploadFile({
                originalname: fileName,
                buffer: zipBuffer,
                mimetype: 'application/zip',
                size: zipBuffer.length
            }, currentPath, userId, username);

            return uploadResult;

        } catch (error) {
            console.error('Compress error:', error);
            throw error;
        }
    }

    /**
     * Extract file
     */
    async extractFile(fileId, extractPath, userId, username) {
        try {
            const file = await this.getFileById(fileId);
            if (!file || !file.name.endsWith('.zip')) throw new Error('Invalid file for extraction');

            // Download zip
            const { data: fileData, error: downError } = await supabase.storage
                .from(STORAGE_BUCKET)
                .download(file.storage_path);

            if (downError) throw downError;

            const buffer = Buffer.from(await fileData.arrayBuffer());
            const zip = new AdmZip(buffer);
            const zipEntries = zip.getEntries();

            // Extract each entry
            for (const entry of zipEntries) {
                if (entry.isDirectory) continue; // Skip folders for flattened structure or implement folder creation logic

                // For simplicity, extract flat to current path or create folder?
                // Creating subfolders recursively is complex.
                // We will extract to extractPath. If entry has valid name.

                const entryName = entry.entryName;
                // Basic implementation: Extract flattened
                const cleanName = path.basename(entryName);

                await this.uploadFile({
                    originalname: cleanName,
                    buffer: entry.getData(),
                    mimetype: 'application/octet-stream', // heuristic
                    size: entry.header.size
                }, extractPath, userId, username);
            }

            return true;
        } catch (error) {
            console.error('Extract error:', error);
            throw error;
        }
    }

    /**
     * Get file content (text)
     */
    async getFileContent(fileId) {
        try {
            const { data: fileRecord, error: dbError } = await supabase
                .from('files')
                .select('*')
                .eq('id', fileId)
                .single();

            if (dbError || !fileRecord) throw new Error('File not found');

            const { data, error } = await supabase.storage
                .from(STORAGE_BUCKET)
                .download(fileRecord.storage_path);

            if (error) throw error;

            return await data.text();
        } catch (error) {
            console.error('Error getting content:', error);
            throw error;
        }
    }

    /**
     * Update file content
     */
    async updateFileContent(fileId, content, userId, username) {
        try {
            // Get current file
            const { data: fileRecord, error: dbError } = await supabase
                .from('files')
                .select('*')
                .eq('id', fileId)
                .single();

            if (dbError || !fileRecord) throw new Error('File not found');

            // Update in storage (overwrite)
            const { error: uploadError } = await supabase.storage
                .from(STORAGE_BUCKET)
                .upload(fileRecord.storage_path, content, {
                    upsert: true,
                    contentType: 'text/plain' // Or detect based on extension? Text content is usually fine as plain/html/etc.
                });

            if (uploadError) throw uploadError;

            // Update size in DB
            const size = Buffer.byteLength(content, 'utf8');
            await supabase
                .from('files')
                .update({ size, updated_at: new Date() })
                .eq('id', fileId);

            // Log activity
            await logService.createLog(userId, 'edit', fileRecord.name, { path: fileRecord.path });
            await discordService.logFileEdit(username, fileRecord.name, fileRecord.path);

            return true;
        } catch (error) {
            console.error('Error updating content:', error);
            throw error;
        }
    }
}

module.exports = new FileService();
