require('dotenv').config();
const express = require('express');
const cookieParser = require('cookie-parser');
const cors = require('cors');
const path = require('path');
const { ensureStorageBucket } = require('./config/database');
const { apiLimiter } = require('./middleware/rateLimiter');

// Import routes
const authRoutes = require('./routes/auth');
const fileRoutes = require('./routes/files');
const terminalRoutes = require('./routes/terminal');
const userRoutes = require('./routes/users');
const logRoutes = require('./routes/logs');

const app = express();
const PORT = process.env.PORT || 3000;

// Middleware
app.use(cors({
    origin: [
        'http://localhost:5500',
        'http://127.0.0.1:5500',
        'http://172.20.10.2:5500',
        process.env.FRONTEND_URL
    ].filter(Boolean),
    credentials: true
}));
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(cookieParser());

// Apply rate limiting to all API routes
app.use('/api', apiLimiter);

// API Routes
app.use('/api/auth', authRoutes);
app.use('/api/files', fileRoutes);
app.use('/api/terminal', terminalRoutes);
app.use('/api/users', userRoutes);
app.use('/api/logs', logRoutes);

// Root endpoint (Dev Helper)
app.get('/', (req, res) => {
    if (process.env.NODE_ENV === 'production') {
        return res.sendFile(path.join(__dirname, '../../frontend/index.html'));
    }
    res.send(`
        <html>
            <body style="font-family: system-ui; display: flex; flex-direction: column; align-items: center; justify-content: center; height: 100vh; background: #111; color: #fff;">
                <h1>🚀 Backend is Running</h1>
                <p>This is the API server (Port ${PORT}).</p>
                <p>To view the application, please visit the frontend server:</p>
                <a href="http://localhost:5500" style="color: #4CAF50; font-size: 1.2rem; text-decoration: none; border: 1px solid #4CAF50; padding: 10px 20px; border-radius: 5px;">
                    Go to Frontend (http://localhost:5500)
                </a>
            </body>
        </html>
    `);
});

// Health check endpoint
app.get('/health', (req, res) => {
    res.json({ status: 'ok', timestamp: new Date().toISOString() });
});

// Serve static files from frontend (if in production)
if (process.env.NODE_ENV === 'production') {
    app.use(express.static(path.join(__dirname, '../../frontend')));

    app.get('*', (req, res) => {
        res.sendFile(path.join(__dirname, '../../frontend/index.html'));
    });
}

// Error handling middleware
app.use((err, req, res, next) => {
    console.error('Error:', err);

    if (err.type === 'entity.too.large') {
        return res.status(413).json({ error: 'File too large' });
    }

    res.status(err.status || 500).json({
        error: err.message || 'Internal server error'
    });
});

// 404 handler
app.use((req, res) => {
    res.status(404).json({ error: 'Not found' });
});

// Initialize server
async function startServer() {
    try {
        // Ensure storage bucket exists
        console.log('Checking Supabase storage bucket...');
        await ensureStorageBucket();

        // Start server
        app.listen(PORT, () => {
            console.log(`\n🚀 File Management System Server`);
            console.log(`━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━`);
            console.log(`📡 Server running on: http://localhost:${PORT}`);
            console.log(`🌍 Environment: ${process.env.NODE_ENV || 'development'}`);
            console.log(`📦 Storage bucket: ${process.env.STORAGE_BUCKET}`);
            console.log(`━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n`);
        });
    } catch (error) {
        console.error('Failed to start server:', error);
        process.exit(1);
    }
}

// Handle graceful shutdown
process.on('SIGTERM', () => {
    console.log('SIGTERM received, shutting down gracefully...');
    process.exit(0);
});

process.on('SIGINT', () => {
    console.log('\nSIGINT received, shutting down gracefully...');
    process.exit(0);
});

// Start the server
startServer();

module.exports = app;
