const authService = require('../services/authService');

/**
 * Authentication middleware
 * Validates session token and attaches user to request
 */
async function authenticate(req, res, next) {
    try {
        // Get token from cookie or Authorization header
        const token = req.cookies?.session_token || req.headers.authorization?.replace('Bearer ', '');

        if (!token) {
            return res.status(401).json({ error: 'Authentication required' });
        }

        // Validate session
        const session = await authService.validateSession(token);

        if (!session || !session.users) {
            return res.status(401).json({ error: 'Invalid or expired session' });
        }

        // Attach user to request
        req.user = session.users;
        req.sessionToken = token;

        next();
    } catch (error) {
        console.error('Authentication error:', error);
        return res.status(401).json({ error: 'Authentication failed' });
    }
}

/**
 * Optional authentication middleware
 * Attaches user if authenticated, but doesn't require it
 */
async function optionalAuth(req, res, next) {
    try {
        const token = req.cookies?.session_token || req.headers.authorization?.replace('Bearer ', '');

        if (token) {
            const session = await authService.validateSession(token);
            if (session && session.users) {
                req.user = session.users;
                req.sessionToken = token;
            }
        }

        next();
    } catch (error) {
        // Continue without authentication
        next();
    }
}

module.exports = {
    authenticate,
    optionalAuth
};
