const fileService = require('./fileService');

class TerminalService {
    constructor() {
        // Track current directory per user session
        this.userDirectories = new Map();
    }

    /**
     * Get current directory for user
     */
    getCurrentDirectory(userId) {
        return this.userDirectories.get(userId) || '/';
    }

    /**
     * Set current directory for user
     */
    setCurrentDirectory(userId, directory) {
        this.userDirectories.set(userId, directory);
    }

    /**
     * Execute a terminal command
     */
    async executeCommand(command, userId, username, permissions) {
        try {
            const parts = command.trim().split(/\s+/);
            const cmd = parts[0].toLowerCase();
            const args = parts.slice(1);

            const currentDir = this.getCurrentDirectory(userId);

            switch (cmd) {
                case 'ls':
                case 'dir':
                    return await this.listDirectory(currentDir, permissions);

                case 'cd':
                    return await this.changeDirectory(args[0], userId, currentDir);

                case 'mkdir':
                    return await this.makeDirectory(args[0], userId, username, currentDir, permissions);

                case 'rm':
                case 'del':
                    return await this.removeFile(args[0], userId, username, currentDir, permissions);

                case 'mv':
                case 'move':
                    return await this.moveFile(args[0], args[1], userId, username, currentDir, permissions);

                case 'cat':
                case 'type':
                    return await this.displayFile(args[0], currentDir, permissions);

                case 'pwd':
                    return { success: true, output: currentDir };

                case 'help':
                    return this.showHelp();

                case 'clear':
                case 'cls':
                    return { success: true, output: '', clear: true };

                default:
                    return { success: false, output: `Command not found: ${cmd}\nType 'help' for available commands.` };
            }
        } catch (error) {
            return { success: false, output: `Error: ${error.message}` };
        }
    }

    /**
     * List directory contents
     */
    async listDirectory(directory, permissions) {
        if (!permissions.read) {
            return { success: false, output: 'Permission denied: read access required' };
        }

        try {
            const files = await fileService.listFiles(directory);

            if (files.length === 0) {
                return { success: true, output: 'Directory is empty' };
            }

            let output = 'Name                          Type      Size\n';
            output += '─'.repeat(60) + '\n';

            files.forEach(file => {
                const name = file.name.padEnd(30);
                const type = file.type.padEnd(10);
                const size = file.type === 'file' ? this.formatFileSize(file.size) : '-';
                output += `${name}${type}${size}\n`;
            });

            return { success: true, output };
        } catch (error) {
            return { success: false, output: `Error listing directory: ${error.message}` };
        }
    }

    /**
     * Change directory
     */
    async changeDirectory(targetDir, userId, currentDir) {
        if (!targetDir) {
            return { success: false, output: 'Usage: cd <directory>' };
        }

        let newDir;
        if (targetDir === '..') {
            // Go up one level
            const parts = currentDir.split('/').filter(p => p);
            parts.pop();
            newDir = '/' + parts.join('/');
            if (newDir === '') newDir = '/';
        } else if (targetDir.startsWith('/')) {
            // Absolute path
            newDir = targetDir;
        } else {
            // Relative path
            newDir = currentDir === '/' ? `/${targetDir}` : `${currentDir}/${targetDir}`;
        }

        // Normalize path
        newDir = newDir.replace(/\/+/g, '/');

        this.setCurrentDirectory(userId, newDir);
        return { success: true, output: `Changed directory to: ${newDir}` };
    }

    /**
     * Make directory
     */
    async makeDirectory(dirName, userId, username, currentDir, permissions) {
        if (!permissions.write) {
            return { success: false, output: 'Permission denied: write access required' };
        }

        if (!dirName) {
            return { success: false, output: 'Usage: mkdir <directory_name>' };
        }

        try {
            await fileService.createFolder(dirName, currentDir, userId, username);
            return { success: true, output: `Directory created: ${dirName}` };
        } catch (error) {
            return { success: false, output: `Error creating directory: ${error.message}` };
        }
    }

    /**
     * Remove file or directory
     */
    async removeFile(fileName, userId, username, currentDir, permissions) {
        if (!permissions.delete) {
            return { success: false, output: 'Permission denied: delete access required' };
        }

        if (!fileName) {
            return { success: false, output: 'Usage: rm <file_name>' };
        }

        try {
            // Find file in current directory
            const files = await fileService.listFiles(currentDir);
            const file = files.find(f => f.name === fileName);

            if (!file) {
                return { success: false, output: `File not found: ${fileName}` };
            }

            await fileService.deleteFile(file.id, userId, username);
            return { success: true, output: `Deleted: ${fileName}` };
        } catch (error) {
            return { success: false, output: `Error deleting file: ${error.message}` };
        }
    }

    /**
     * Move file
     */
    async moveFile(source, destination, userId, username, currentDir, permissions) {
        if (!permissions.move) {
            return { success: false, output: 'Permission denied: move access required' };
        }

        if (!source || !destination) {
            return { success: false, output: 'Usage: mv <source> <destination>' };
        }

        try {
            // Find source file
            const files = await fileService.listFiles(currentDir);
            const file = files.find(f => f.name === source);

            if (!file) {
                return { success: false, output: `File not found: ${source}` };
            }

            // Determine destination path
            let destPath = destination.startsWith('/') ? destination : `${currentDir}/${destination}`;
            destPath = destPath.replace(/\/+/g, '/');

            await fileService.moveFile(file.id, destPath, userId, username);
            return { success: true, output: `Moved ${source} to ${destPath}` };
        } catch (error) {
            return { success: false, output: `Error moving file: ${error.message}` };
        }
    }

    /**
     * Display file contents (for text files)
     */
    async displayFile(fileName, currentDir, permissions) {
        if (!permissions.read) {
            return { success: false, output: 'Permission denied: read access required' };
        }

        if (!fileName) {
            return { success: false, output: 'Usage: cat <file_name>' };
        }

        try {
            // Find file in current directory
            const files = await fileService.listFiles(currentDir);
            const file = files.find(f => f.name === fileName);

            if (!file) {
                return { success: false, output: `File not found: ${fileName}` };
            }

            if (file.type === 'folder') {
                return { success: false, output: `${fileName} is a directory` };
            }

            // For now, just show file info
            let output = `File: ${file.name}\n`;
            output += `Size: ${this.formatFileSize(file.size)}\n`;
            output += `Owner: ${file.users?.username || 'Unknown'}\n`;
            output += `Created: ${new Date(file.created_at).toLocaleString()}\n`;
            output += `\nNote: File preview not available in terminal. Use the file manager to view contents.`;

            return { success: true, output };
        } catch (error) {
            return { success: false, output: `Error displaying file: ${error.message}` };
        }
    }

    /**
     * Show help
     */
    showHelp() {
        const output = `Available Commands:
─────────────────────────────────────────────────────────
ls, dir          List files and folders in current directory
cd <path>        Change directory
pwd              Print working directory
mkdir <name>     Create a new folder
rm <name>        Delete a file or folder
mv <src> <dst>   Move a file or folder
cat <file>       Display file information
help             Show this help message
clear, cls       Clear the terminal screen
─────────────────────────────────────────────────────────
Note: All commands respect your user permissions.`;

        return { success: true, output };
    }

    /**
     * Format file size
     */
    formatFileSize(bytes) {
        if (bytes === 0) return '0 B';
        const k = 1024;
        const sizes = ['B', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
    }
}

module.exports = new TerminalService();
