const express = require('express');
const router = express.Router();
const authService = require('../services/authService');
const discordService = require('../services/discordService');
const { authenticate } = require('../middleware/auth');
const { requireAdmin } = require('../middleware/permissions');

/**
 * GET /api/users
 * Get all users (admin only)
 */
router.get('/', authenticate, requireAdmin, async (req, res) => {
    try {
        const users = await authService.getAllUsers();
        res.json({ success: true, users });
    } catch (error) {
        console.error('Get users error:', error);
        res.status(500).json({ error: 'Failed to get users' });
    }
});

/**
 * POST /api/users
 * Create a new user (admin only)
 */
router.post('/', authenticate, requireAdmin, async (req, res) => {
    try {
        const { username, password, role, permissions } = req.body;

        if (!username || !password) {
            return res.status(400).json({ error: 'Username and password are required' });
        }

        if (username.length < 3) {
            return res.status(400).json({ error: 'Username must be at least 3 characters' });
        }

        if (password.length < 6) {
            return res.status(400).json({ error: 'Password must be at least 6 characters' });
        }

        const user = await authService.createUser(username, password, role || 'staff', permissions);

        // Log user creation
        await discordService.logUserCreate(req.user.username, username, user.role);

        const { password_hash, ...userWithoutPassword } = user;
        res.status(201).json({ success: true, user: userWithoutPassword });
    } catch (error) {
        console.error('Create user error:', error);
        res.status(400).json({ error: error.message || 'Failed to create user' });
    }
});

/**
 * PUT /api/users/:id/permissions
 * Update user permissions (admin only)
 */
router.put('/:id/permissions', authenticate, requireAdmin, async (req, res) => {
    try {
        const { id } = req.params;
        const { permissions } = req.body;

        if (!permissions) {
            return res.status(400).json({ error: 'Permissions object is required' });
        }

        // Get target user info for logging
        const targetUser = await authService.getUserById(id);
        if (!targetUser) {
            return res.status(404).json({ error: 'User not found' });
        }

        const user = await authService.updatePermissions(id, permissions);

        // Log permission change
        await discordService.logPermissionChange(req.user.username, targetUser.username, permissions);

        res.json({ success: true, user });
    } catch (error) {
        console.error('Update permissions error:', error);
        res.status(500).json({ error: error.message || 'Failed to update permissions' });
    }
});

/**
 * DELETE /api/users/:id
 * Delete a user (admin only)
 */
router.delete('/:id', authenticate, requireAdmin, async (req, res) => {
    try {
        const { id } = req.params;

        // Prevent admin from deleting themselves
        if (id === req.user.id) {
            return res.status(400).json({ error: 'Cannot delete your own account' });
        }

        await authService.deleteUser(id);
        res.json({ success: true, message: 'User deleted successfully' });
    } catch (error) {
        console.error('Delete user error:', error);
        res.status(500).json({ error: 'Failed to delete user' });
    }
});

/**
 * GET /api/users/:id
 * Get user by ID (admin only)
 */
router.get('/:id', authenticate, requireAdmin, async (req, res) => {
    try {
        const { id } = req.params;
        const user = await authService.getUserById(id);

        if (!user) {
            return res.status(404).json({ error: 'User not found' });
        }

        res.json({ success: true, user });
    } catch (error) {
        console.error('Get user error:', error);
        res.status(500).json({ error: 'Failed to get user' });
    }
});

module.exports = router;
