const express = require('express');
const router = express.Router();
const multer = require('multer');
const fileService = require('../services/fileService');
const { authenticate } = require('../middleware/auth');
const { checkPermission } = require('../middleware/permissions');
const { uploadLimiter } = require('../middleware/rateLimiter');

// Configure multer for file uploads (memory storage)
const upload = multer({
    storage: multer.memoryStorage(),
    limits: {
        fileSize: 50 * 1024 * 1024 // 50MB limit
    }
});

/**
 * GET /api/files/trash/list
 * List trash files
 */
router.get('/trash/list', authenticate, async (req, res) => {
    try {
        console.log('GET /trash/list called by:', req.user.username);
        const files = await fileService.listTrash(req.user.id);
        console.log('Trash files found:', files.length);
        res.json({ success: true, files });
    } catch (error) {
        console.error('List trash error:', error);
        res.status(500).json({ error: 'Failed to list trash' });
    }
});

/**
 * DELETE /api/files/trash/empty
 * Empty trash
 */
router.delete('/trash/empty', authenticate, checkPermission('empty_trash'), async (req, res) => {
    try {
        const files = await fileService.listTrash(req.user.id);
        for (const file of files) {
            await fileService.permanentDelete(file.id, req.user.id, req.user.username);
        }
        res.json({ success: true });
    } catch (error) {
        console.error('Empty trash error:', error);
        res.status(500).json({ error: 'Failed to empty trash' });
    }
});

/**
 * PUT /api/files/restore/:id
 * Restore file from trash
 */
router.put('/restore/:id', authenticate, checkPermission('delete'), async (req, res) => {
    try {
        const { id } = req.params;
        await fileService.restoreFile(id, req.user.id, req.user.username);
        res.json({ success: true, message: 'File restored successfully' });
    } catch (error) {
        console.error('Restore error:', error);
        res.status(500).json({ error: error.message || 'Failed to restore file' });
    }
});

/**
 * GET /api/files
 * List files and folders
 */
router.get('/', authenticate, checkPermission('read'), async (req, res) => {
    try {
        const { path = '/' } = req.query;
        const files = await fileService.listFiles(path, req.user.id);
        res.json({ success: true, files });
    } catch (error) {
        console.error('List files error:', error);
        res.status(500).json({ error: 'Failed to list files' });
    }
});

/**
 * POST /api/files/upload
 * Upload a file
 */
router.post('/upload', authenticate, checkPermission('write'), uploadLimiter, upload.single('file'), async (req, res) => {
    try {
        if (!req.file) {
            return res.status(400).json({ error: 'No file provided' });
        }

        const { path = '/' } = req.body;
        const file = await fileService.uploadFile(req.file, path, req.user.id, req.user.username);

        res.status(201).json({ success: true, file });
    } catch (error) {
        console.error('Upload error:', error);
        res.status(500).json({ error: error.message || 'Failed to upload file' });
    }
});

/**
 * GET /api/files/download/:id
 * Download a file
 */
router.get('/download/:id', authenticate, checkPermission('read'), async (req, res) => {
    try {
        const { id } = req.params;
        const result = await fileService.downloadFile(id);

        // Convert blob to buffer
        const buffer = Buffer.from(await result.data.arrayBuffer());

        res.setHeader('Content-Disposition', `attachment; filename="${result.name}"`);
        res.setHeader('Content-Type', 'application/octet-stream');
        res.setHeader('Content-Length', buffer.length);
        res.send(buffer);
    } catch (error) {
        console.error('Download error:', error);
        res.status(500).json({ error: error.message || 'Failed to download file' });
    }
});

/**
 * PUT /api/files/rename/:id
 * Rename a file or folder
 */
router.put('/rename/:id', authenticate, checkPermission('edit'), async (req, res) => {
    try {
        const { id } = req.params;
        const { newName } = req.body;

        if (!newName) {
            return res.status(400).json({ error: 'New name is required' });
        }

        const file = await fileService.renameFile(id, newName, req.user.id, req.user.username);
        res.json({ success: true, file });
    } catch (error) {
        console.error('Rename error:', error);
        res.status(500).json({ error: error.message || 'Failed to rename file' });
    }
});

/**
 * PUT /api/files/move/:id
 * Move a file or folder
 */
router.put('/move/:id', authenticate, checkPermission('move'), async (req, res) => {
    try {
        const { id } = req.params;
        const { newPath } = req.body;

        if (!newPath) {
            return res.status(400).json({ error: 'New path is required' });
        }

        const file = await fileService.moveFile(id, newPath, req.user.id, req.user.username);
        res.json({ success: true, file });
    } catch (error) {
        console.error('Move error:', error);
        res.status(500).json({ error: error.message || 'Failed to move file' });
    }
});

/**
 * DELETE /api/files/:id
 * Delete a file or folder
 */
router.delete('/:id', authenticate, checkPermission('delete'), async (req, res) => {
    try {
        const { id } = req.params;
        await fileService.deleteFile(id, req.user.id, req.user.username);
        res.json({ success: true, message: 'File deleted successfully' });
    } catch (error) {
        console.error('Delete error:', error);
        res.status(500).json({ error: error.message || 'Failed to delete file' });
    }
});

/**
 * POST /api/files/folder
 * Create a new folder
 */
router.post('/folder', authenticate, checkPermission('write'), async (req, res) => {
    try {
        const { name, path = '/' } = req.body;

        if (!name) {
            return res.status(400).json({ error: 'Folder name is required' });
        }

        const folder = await fileService.createFolder(name, path, req.user.id, req.user.username);
        res.status(201).json({ success: true, folder });
    } catch (error) {
        console.error('Create folder error:', error);
        res.status(500).json({ error: error.message || 'Failed to create folder' });
    }
});

/**
 * GET /api/files/search
 * Search files by name
 */
router.get('/search', authenticate, checkPermission('read'), async (req, res) => {
    try {
        const { q } = req.query;

        if (!q) {
            return res.status(400).json({ error: 'Search query is required' });
        }

        const files = await fileService.searchFiles(q, req.user.id);
        res.json({ success: true, files });
    } catch (error) {
        console.error('Search error:', error);
        res.status(500).json({ error: 'Failed to search files' });
    }
});

/**
 * GET /api/files/stats
 * Get storage statistics
 */
router.get('/stats', authenticate, async (req, res) => {
    try {
        const stats = await fileService.getStorageStats(req.user.id);
        res.json({ success: true, stats });
    } catch (error) {
        console.error('Stats error:', error);
        res.status(500).json({ error: 'Failed to get storage stats' });
    }
});

/**
 * GET /api/files/info/:id
 * Get file information
 */
router.get('/info/:id', authenticate, checkPermission('read'), async (req, res) => {
    try {
        const { id } = req.params;
        const file = await fileService.getFileById(id);
        res.json({ success: true, file });
    } catch (error) {
        console.error('Get file info error:', error);
        res.status(500).json({ error: 'Failed to get file info' });
    }
});

/**
 * POST /api/files/copy/:id
 * Copy a file
 */
router.post('/copy/:id', authenticate, checkPermission('write'), async (req, res) => {
    try {
        const { id } = req.params;
        const { destPath } = req.body;

        if (!destPath) {
            return res.status(400).json({ error: 'Destination path is required' });
        }

        const file = await fileService.copyFile(id, destPath, req.user.id, req.user.username);
        res.status(201).json({ success: true, file });
    } catch (error) {
        console.error('Copy error:', error);
        res.status(500).json({ error: error.message || 'Failed to copy file' });
    }
});



/**
 * POST /api/files/compress
 * Compress files
 */
router.post('/compress', authenticate, checkPermission('write'), async (req, res) => {
    try {
        const { fileIds, archiveName, currentPath } = req.body;
        await fileService.compressFiles(fileIds, archiveName, currentPath, req.user.id, req.user.username);
        res.json({ success: true });
    } catch (error) {
        console.error('Compress error:', error);
        res.status(500).json({
            error: 'Failed to compress files',
            details: error.message,
            stack: error.stack
        });
    }
});

/**
 * POST /api/files/extract
 * Extract file
 */
router.post('/extract', authenticate, checkPermission('write'), async (req, res) => {
    try {
        const { fileId, extractPath } = req.body;
        await fileService.extractFile(fileId, extractPath, req.user.id, req.user.username);
        res.json({ success: true });
    } catch (error) {
        console.error('Extract error:', error);
        res.status(500).json({ error: 'Failed to extract file' });
    }
});

/**
 * GET /api/files/content/:id
 * Get file content
 */
router.get('/content/:id', authenticate, checkPermission('read'), async (req, res) => {
    try {
        const { id } = req.params;
        const content = await fileService.getFileContent(id);
        res.json({ success: true, content });
    } catch (error) {
        console.error('Get content error:', error);
        res.status(500).json({ error: 'Failed to get file content' });
    }
});

/**
 * PUT /api/files/content/:id
 * Update file content
 */
router.put('/content/:id', authenticate, checkPermission('edit'), async (req, res) => {
    try {
        const { id } = req.params;
        const { content } = req.body;

        if (content === undefined) {
            return res.status(400).json({ error: 'Content is required' });
        }

        await fileService.updateFileContent(id, content, req.user.id, req.user.username);
        res.json({ success: true, message: 'File saved successfully' });
    } catch (error) {
        console.error('Update content error:', error);
        res.status(500).json({ error: 'Failed to update file content' });
    }
});

module.exports = router;
