/**
 * Permission checking middleware
 */

/**
 * Check if user has specific permission
 */
function checkPermission(permissionName) {
    return (req, res, next) => {
        if (!req.user) {
            return res.status(401).json({ error: 'Authentication required' });
        }

        // Admins have all permissions
        if (req.user.role === 'admin') {
            return next();
        }

        // Check if user has the required permission
        if (!req.user.permissions || !req.user.permissions[permissionName]) {
            return res.status(403).json({ error: `Permission denied: ${permissionName} access required` });
        }

        next();
    };
}

/**
 * Check if user is admin
 */
function requireAdmin(req, res, next) {
    if (!req.user) {
        return res.status(401).json({ error: 'Authentication required' });
    }

    if (req.user.role !== 'admin') {
        return res.status(403).json({ error: 'Admin access required' });
    }

    next();
}

/**
 * Check if user has any of the specified roles
 */
function requireRole(...roles) {
    return (req, res, next) => {
        if (!req.user) {
            return res.status(401).json({ error: 'Authentication required' });
        }

        if (!roles.includes(req.user.role)) {
            return res.status(403).json({ error: 'Insufficient privileges' });
        }

        next();
    };
}

module.exports = {
    checkPermission,
    requireAdmin,
    requireRole
};
