// File Manager Logic
class FileManager {
    constructor() {
        this.currentPath = '/';
        this.files = [];
        this.selectedFiles = new Set();
        this.isTrashMode = false;
        this.init();
    }

    async init() {
        await this.loadFiles();
        await this.loadStats();
        this.setupEventListeners();
    }

    async loadFiles() {
        try {
            showLoading();
            this.isTrashMode = false;
            const response = await api.listFiles(this.currentPath);
            this.files = response.files || [];
            this.updateToolbar();
            this.renderFiles();
            this.renderBreadcrumbs();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    async loadStats() {
        try {
            const response = await api.getStorageStats();
            this.renderStats(response.stats);

            // Update media count (approximate from current view per user request)
            if (this.files) {
                const mediaExts = ['pdf', 'jpg', 'jpeg', 'png', 'gif', 'bmp', 'svg', 'webp'];
                const mediaCount = this.files.filter(f => f.type === 'file' && mediaExts.includes(f.name.split('.').pop().toLowerCase())).length;
                const mediaEl = document.getElementById('total-media-count');
                if (mediaEl) mediaEl.textContent = mediaCount;
            }
        } catch (error) {
            console.error('Failed to load stats:', error);
        }
    }

    renderStats(stats) {
        // Dynamic Unit Formatting
        const formatSize = (bytes) => {
            if (bytes === 0) return '0 B';
            const k = 1024;
            const sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        };

        const maxBytes = 500 * 1024 * 1024 * 1024; // 500GB
        const usedStr = formatSize(stats.totalSize);
        const totalStr = formatSize(maxBytes);
        const percentage = Math.min((stats.totalSize / maxBytes) * 100, 100);

        const usedEl = document.getElementById('storage-used');
        if (usedEl) usedEl.textContent = usedStr;

        const totalEl = document.getElementById('storage-total');
        if (totalEl) totalEl.textContent = `/ ${totalStr}`;

        const filesEl = document.getElementById('total-files-count');
        if (filesEl) filesEl.textContent = stats.totalFiles;

        const foldersEl = document.getElementById('total-folders-count');
        if (foldersEl) foldersEl.textContent = stats.totalFolders;

        // Update progress circle
        const circle = document.querySelector('.storage-progress circle:last-child');
        if (circle) {
            const circumference = 339.292; // 2 * PI * 54
            const offset = circumference - (percentage / 100) * circumference;
            circle.style.strokeDashoffset = offset;
        }
    }

    renderBreadcrumbs() {
        const breadcrumbs = document.getElementById('breadcrumbs');
        if (!breadcrumbs) return;

        // Trash Mode Breadcrumb
        if (this.isTrashMode) {
            breadcrumbs.innerHTML = `
                <div class="breadcrumb-container" style="display: flex; align-items: center; background: rgba(0,0,0,0.2); padding: 5px 15px; border-radius: 4px;">
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <span class="breadcrumb-item"><i class="fas fa-trash"></i> Trash</span>
                    </div>
                </div>
            `;
            return;
        }

        // Standard Mode Breadcrumb
        const parts = this.currentPath.split('/').filter(p => p);

        let html = `
      <div class="breadcrumb-container" style="display: flex; align-items: center; background: rgba(0,0,0,0.2); padding: 5px 10px; border-radius: 4px;">
        <span class="breadcrumb-item">
          <a href="#" class="breadcrumb-link" data-path="/"><i class="fas fa-home"></i> Home</a>
        </span>
    `;

        let path = '';
        parts.forEach((part, index) => {
            path += '/' + part;
            html += `
        <span class="breadcrumb-separator" style="margin: 0 8px; opacity: 0.5;">/</span>
        <span class="breadcrumb-item">
          <a href="#" class="breadcrumb-link" data-path="${path}">${escapeHtml(part)}</a>
        </span>
      `;
        });

        html += `</div>`;
        breadcrumbs.innerHTML = html;

        // Add click listeners
        breadcrumbs.querySelectorAll('.breadcrumb-link').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                this.navigateTo(link.dataset.path);
            });
        });
    }

    renderFiles() {
        const container = document.getElementById('files-container');
        if (!container) return;

        if (this.files.length === 0) {
            const emptyIcon = this.isTrashMode ? 'trash-alt' : 'folder-open';
            const emptyTitle = this.isTrashMode ? 'Trash is Empty' : 'No files here';
            const emptyDesc = this.isTrashMode ? 'Deleted files appear here.' : 'Drag and drop files to upload or create a new folder';

            container.innerHTML = `
        <tr>
          <td colspan="5">
            <div class="empty-state">
              <div class="empty-icon"><i class="fas fa-${emptyIcon}"></i></div>
              <h3 class="empty-title" style="color: var(--text-primary)">${emptyTitle}</h3>
              <p class="empty-description">${emptyDesc}</p>
            </div>
          </td>
        </tr>
      `;
            return;
        }

        const html = this.files.map(file => `
      <tr data-file-id="${file.id}" class="file-row" draggable="true">
        <td>
          <div style="display: flex; align-items: center; gap: 12px;">
            <div class="file-icon">${getFileIcon(file.name, file.type)}</div>
            <span class="file-name" style="color: var(--text-primary)">${escapeHtml(file.name)}</span>
          </div>
        </td>
        <td>${file.type === 'folder' ? 'Folder' : 'File'}</td>
        <td>${file.type === 'file' ? formatFileSize(file.size) : '-'}</td>
        <td>${this.isTrashMode && file.deleted_at ? formatDate(file.deleted_at) : formatDate(file.updated_at || file.created_at)}</td>
        <td>
          <button class="btn btn-sm btn-icon" onclick="fileManager.showContextMenu(event, '${file.id}')">
            <i class="fas fa-ellipsis-v"></i>
          </button>
        </td>
      </tr>
    `).join('');

        container.innerHTML = html;

        // Add click/drag listeners
        container.querySelectorAll('.file-row').forEach(row => {
            if (!this.isTrashMode) {
                // Drag start (only if not in trash)
                row.addEventListener('dragstart', (e) => {
                    e.dataTransfer.setData('text/plain', row.dataset.fileId);
                    e.dataTransfer.effectAllowed = 'move';
                });
            }

            // Double click to open
            row.addEventListener('dblclick', () => {
                const fileId = row.dataset.fileId;
                const file = this.files.find(f => f.id === fileId);

                if (this.isTrashMode) return; // Cannot open files in trash

                if (file && file.type === 'folder') {
                    this.navigateTo(file.path === '/' ? `/${file.name}` : `${file.path}/${file.name}`);
                } else if (file && file.type === 'file') {
                    // Check if editable
                    const editable = [
                        'html', 'htm', 'css', 'scss', 'less',
                        'js', 'jsx', 'ts', 'tsx', 'json',
                        'txt', 'md', 'xml', 'svg',
                        'sql', 'py', 'rb', 'php', 'java', 'c', 'cpp', 'h', 'cs', 'go', 'rs', 'swift', 'kt', 'lua',
                        'sh', 'bat', 'ps1',
                        'yml', 'yaml', 'ini', 'env', 'config', 'conf', 'properties', 'toml',
                        'log', 'rtf'
                    ];
                    const ext = file.name.split('.').pop().toLowerCase();

                    if (editable.includes(ext) || file.name.startsWith('.')) {
                        console.log('Opening editor for file:', fileId);
                        this.openEditor(fileId);
                    } else {
                        this.downloadFile(fileId);
                    }
                }
            });

            // Right click context menu
            row.addEventListener('contextmenu', (e) => {
                e.preventDefault();
                this.showContextMenu(e, row.dataset.fileId);
            });
        });
    }

    navigateTo(path) {
        this.currentPath = path.replace(/\/+/g, '/') || '/';
        this.loadFiles();
    }

    async loadTrashFiles() {
        try {
            const response = await api.listTrash();
            this.files = response.files || [];
            this.renderFiles();
        } catch (error) {
            console.error('Failed to load trash:', error);
            showToast('Failed to load trash files', 'error');
        }
    }

    async navigateToTrash() {
        try {
            showLoading();
            this.isTrashMode = true;
            this.currentPath = '/'; // Reset visually mostly
            this.updateToolbar();
            await this.loadTrashFiles();
            this.renderBreadcrumbs();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    async uploadFile(file) {
        if (this.isTrashMode) {
            showToast('Cannot upload to trash', 'warning');
            return;
        }
        try {
            showLoading();
            showToast(`Uploading ${file.name}...`, 'info');
            await api.uploadFile(file, this.currentPath);
            showToast('File uploaded successfully', 'success');
            await this.loadFiles();
            await this.loadStats();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    async createFolder() {
        if (this.isTrashMode) return;
        const name = await Modal.prompt('Enter folder name:', '', 'New Folder');
        if (!name) return;

        try {
            showLoading();
            const path = this.currentPath;
            await api.createFolder(name, path);
            showToast(`Folder "${name}" created`, 'success');
            // Force reload to ensure it appears
            await this.loadFiles();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    async renameFile(fileId) {
        if (this.isTrashMode) return;
        const file = this.files.find(f => f.id === fileId);
        if (!file) return;

        const newName = await Modal.prompt('Enter new name:', file.name, 'Rename File');
        if (!newName || newName === file.name) return;

        try {
            showLoading();
            await api.renameFile(fileId, newName);
            showToast('Renamed successfully', 'success');
            await this.loadFiles();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    async deleteFile(fileId) {
        const file = this.files.find(f => f.id === fileId);
        if (!file) return;

        const msg = this.isTrashMode
            ? `Permanently delete "${file.name}"? This cannot be undone.`
            : `Move "${file.name}" to Trash?`;

        if (!await Modal.confirm(msg, 'Delete File', { type: 'danger' })) return;

        try {
            showLoading();

            if (this.isTrashMode) {
                // For now, user must use "Empty Trash" or we need a specific permanent delete API
                showToast('Please use "Empty Trash" to delete permanently', 'warning');
                return;
            }

            await api.deleteFile(fileId);
            showToast('Moved to Trash', 'success');

            // Refresh based on current mode
            if (this.isTrashMode) {
                await this.loadTrashFiles();
            } else {
                await this.loadFiles();
            }

            await this.loadStats();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    async restoreFile(fileId) {
        try {
            showLoading();
            await api.restoreFile(fileId);
            showToast('Restored successfully', 'success');
            await this.loadTrashFiles(); // Refresh trash list
            await this.loadStats();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    async emptyTrash() {
        if (!await Modal.confirm('Permanently delete all items in Trash? This cannot be undone.', 'Empty Trash', { type: 'danger' })) return;

        try {
            showLoading();
            await api.emptyTrash();
            showToast('Trash emptied', 'success');
            await this.loadTrashFiles();
            await this.loadStats();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    async moveFile(fileId) {
        if (this.isTrashMode) return;
        const file = this.files.find(f => f.id === fileId);
        if (!file) return;

        const newPath = await Modal.prompt('Enter destination path (e.g., /documents):', this.currentPath, 'Move File');
        if (!newPath || newPath === this.currentPath) return;

        try {
            showLoading();
            await api.moveFile(fileId, newPath);
            showToast('Moved successfully', 'success');
            await this.loadFiles();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    async copyFile(fileId) {
        if (this.isTrashMode) return;
        const file = this.files.find(f => f.id === fileId);
        if (!file) return;

        const destPath = await Modal.prompt('Enter destination path (e.g., /documents):', this.currentPath, 'Copy File');
        if (!destPath) return;

        try {
            showLoading();
            await api.copyFile(fileId, destPath);
            showToast('Copied successfully', 'success');
            await this.loadFiles();
            await this.loadStats();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    async compressFile(fileId) {
        if (this.isTrashMode) return;
        const file = this.files.find(f => f.id === fileId);
        if (!file) return;

        const archiveName = await Modal.prompt('Enter archive name:', file.name + '.zip', 'Compress File');
        if (!archiveName) return;

        try {
            showLoading();
            // Support single file compress for now
            await api.compressFiles([fileId], archiveName, this.currentPath);
            showToast('Compressed successfully', 'success');
            await this.loadFiles();
            await this.loadStats();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    async extractFile(fileId) {
        if (this.isTrashMode) return;
        const file = this.files.find(f => f.id === fileId);
        if (!file) return;

        // Extract to current path?
        const extractPath = await Modal.prompt('Extract to:', this.currentPath, 'Extract File');
        if (extractPath === null) return;

        try {
            showLoading();
            await api.extractFile(fileId, extractPath || this.currentPath);
            showToast('Extracted successfully', 'success');
            await this.loadFiles();
            await this.loadStats();
        } catch (error) {
            showToast(error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    downloadFile(fileId) {
        if (this.isTrashMode) return;
        api.downloadFile(fileId);
    }

    showContextMenu(event, fileId) {
        event.preventDefault();
        event.stopPropagation();

        const file = this.files.find(f => f.id === fileId);
        if (!file) return;

        const user = getCurrentUser();
        const menu = document.getElementById('context-menu');

        let html = '';

        if (this.isTrashMode) {
            // Trash Context Menu
            if (user.permissions.restore) {
                html += `<div class="context-menu-item" onclick="fileManager.restoreFile('${fileId}')"><i class="fas fa-undo"></i> Restore</div>`;
            } else {
                html += `<div class="context-menu-item disabled" style="opacity: 0.5; cursor: not-allowed;"><i class="fas fa-undo"></i> Restore (No Permission)</div>`;
            }
            // html += `<div class="context-menu-item danger" onclick="fileManager.deleteFile('${fileId}')"><i class="fas fa-trash"></i> Delete Forever</div>`; 
            // Disabled single delete for now as explained
        } else {
            // Normal Context Menu
            if (file.type === 'folder') {
                html += `<div class="context-menu-item" onclick="fileManager.navigateTo('${file.path === '/' ? '/' + file.name : file.path + '/' + file.name}')"><i class="fas fa-folder-open"></i> Open</div>`;
            } else {
                const editable = [
                    'html', 'htm', 'css', 'scss', 'less',
                    'js', 'jsx', 'ts', 'tsx', 'json',
                    'txt', 'md', 'xml', 'svg',
                    'sql', 'py', 'rb', 'php', 'java', 'c', 'cpp', 'h', 'cs', 'go', 'rs', 'swift', 'kt', 'lua',
                    'sh', 'bat', 'ps1',
                    'yml', 'yaml', 'ini', 'env', 'config', 'conf', 'properties', 'toml',
                    'log', 'rtf'
                ];
                const ext = file.name.split('.').pop().toLowerCase();

                if (editable.includes(ext) || file.name.startsWith('.')) {
                    html += `<div class="context-menu-item" onclick="fileManager.openEditor('${fileId}')"><i class="fas fa-edit"></i> Edit</div>`;
                }

                html += `<div class="context-menu-item" onclick="fileManager.downloadFile('${fileId}')"><i class="fas fa-download"></i> Download</div>`;
            }

            if (user.permissions.edit) {
                html += `<div class="context-menu-item" onclick="fileManager.renameFile('${fileId}')"><i class="fas fa-edit"></i> Rename</div>`;
            }

            if (user.permissions.move) {
                html += `<div class="context-menu-item" onclick="fileManager.moveFile('${fileId}')"><i class="fas fa-arrow-right"></i> Move</div>`;
                html += `<div class="context-menu-item" onclick="fileManager.copyFile('${fileId}')"><i class="fas fa-copy"></i> Copy</div>`;
            }

            if (user.permissions.write) {
                html += `<div class="context-menu-divider"></div>`;
                html += `<div class="context-menu-item" onclick="fileManager.compressFile('${fileId}')"><i class="fas fa-file-archive"></i> Compress</div>`;
                if (file.name.endsWith('.zip') || file.name.endsWith('.tar.gz')) {
                    html += `<div class="context-menu-item" onclick="fileManager.extractFile('${fileId}')"><i class="fas fa-box-open"></i> Extract</div>`;
                }
            }

            if (user.permissions.delete) {
                html += `<div class="context-menu-divider"></div>`;
                html += `<div class="context-menu-item danger" onclick="fileManager.deleteFile('${fileId}')"><i class="fas fa-trash-alt"></i> Delete</div>`;
            }
        }

        menu.innerHTML = html;
        menu.style.display = 'block';

        let x = event.pageX;
        let y = event.pageY;

        if (x + 200 > window.innerWidth) x -= 200;

        menu.style.left = x + 'px';
        menu.style.top = y + 'px';
    }

    setupEventListeners() {
        // File upload button
        const uploadInput = document.getElementById('file-upload');
        if (uploadInput) {
            uploadInput.addEventListener('change', (e) => {
                const file = e.target.files[0];
                if (file) {
                    this.uploadFile(file);
                    e.target.value = '';
                }
            });
        }

        // Drag and drop for upload (content area)
        const contentArea = document.querySelector('.content-area');
        if (contentArea) {
            contentArea.addEventListener('dragover', (e) => {
                e.preventDefault();
                e.dataTransfer.dropEffect = 'copy';
                contentArea.style.border = '2px dashed var(--primary-color)';
            });

            contentArea.addEventListener('dragleave', (e) => {
                contentArea.style.border = 'none';
            });

            contentArea.addEventListener('drop', (e) => {
                e.preventDefault();
                contentArea.style.border = 'none';

                if (this.isTrashMode) return;

                // Handle file drop (upload)
                if (e.dataTransfer.files.length > 0) {
                    const file = e.dataTransfer.files[0];
                    this.uploadFile(file);
                }
            });
        }

        // Drag to trash (sidebar item)
        const trashItem = document.querySelector('.drop-target-trash');
        if (trashItem) {
            trashItem.addEventListener('dragover', (e) => {
                e.preventDefault(); // Allow drop
                e.dataTransfer.dropEffect = 'move';
                trashItem.style.background = 'rgba(239, 68, 68, 0.2)'; // Helper visual
            });

            trashItem.addEventListener('dragleave', (e) => {
                trashItem.style.background = '';
            });

            trashItem.addEventListener('drop', (e) => {
                e.preventDefault();
                trashItem.style.background = '';
                const fileId = e.dataTransfer.getData('text/plain');
                if (fileId && !this.isTrashMode) {
                    this.deleteFile(fileId);
                }
            });
        }

        // Global click to close context menu
        document.addEventListener('click', () => {
            const menu = document.getElementById('context-menu');
            if (menu) menu.style.display = 'none';
        });

        // Search
        const searchInput = document.getElementById('search-input');
        if (searchInput) {
            searchInput.addEventListener('input', debounce(async (e) => {
                const query = e.target.value.trim();
                if (query.length > 0) {
                    try {
                        const response = await api.searchFiles(query);
                        this.files = response.files || [];
                        this.renderFiles();
                    } catch (error) {
                        showToast(error.message, 'error');
                    }
                } else {
                    this.loadFiles();
                }
            }, 300));
        }
    }
    updateToolbar() {
        const user = getCurrentUser();
        const uploadBtn = document.getElementById('upload-btn');
        const newFolderBtn = document.getElementById('new-folder-btn');
        const emptyTrashBtn = document.getElementById('empty-trash-btn');
        const sectionTitle = document.getElementById('section-title');
        const dateHeader = document.getElementById('header-date');

        if (this.isTrashMode) {
            if (sectionTitle) sectionTitle.textContent = 'Trash';
            if (dateHeader) dateHeader.textContent = 'Date Deleted';
            // Trash Mode: Show 'Empty Trash' (if perm), Hide others
            if (uploadBtn) uploadBtn.style.display = 'none';
            if (newFolderBtn) newFolderBtn.style.display = 'none';

            if (emptyTrashBtn) {
                emptyTrashBtn.style.display = user.permissions.empty_trash ? 'inline-flex' : 'none';
            }
        } else {
            if (sectionTitle) sectionTitle.textContent = 'Recently Modified';
            if (dateHeader) dateHeader.textContent = 'Last Modified';
            // Normal Mode: Hide 'Empty Trash', Show others (if perm)
            if (emptyTrashBtn) emptyTrashBtn.style.display = 'none';

            if (uploadBtn) {
                uploadBtn.style.display = user.permissions.write ? 'inline-flex' : 'none';
            }
            if (newFolderBtn) {
                newFolderBtn.style.display = user.permissions.write ? 'inline-flex' : 'none';
            }
        }
    }
    // Editor Methods
    async openEditor(fileId) {
        try {
            showLoading();
            this.currentEditingFileId = fileId;
            const modal = document.getElementById('editor-modal');
            const editorDiv = document.getElementById('editor');

            // Initialize Ace if not already
            if (!this.editor) {
                this.editor = ace.edit("editor");
                this.editor.setTheme("ace/theme/vs_dark");
                this.editor.setFontSize(14);
                this.editor.session.setTabSize(4);
                this.editor.session.setUseWrapMode(true); // Enable wrapping
                this.editor.setOptions({
                    fontFamily: "JetBrains Mono, monospace"
                });

                // Save on Ctrl+S
                this.editor.commands.addCommand({
                    name: 'save',
                    bindKey: { win: 'Ctrl-S', mac: 'Command-S' },
                    exec: () => this.saveEditorContent(),
                    readOnly: true
                });
            }

            // Get File Info & Content
            const [infoResponse, contentResponse] = await Promise.all([
                api.getFileInfo(fileId),
                api.getFileContent(fileId)
            ]);

            const file = infoResponse.file;
            const content = contentResponse.content;

            document.getElementById('editor-filename').textContent = `Editing: ${file.name}`;
            document.getElementById('editor-status').textContent = '';

            // Set Mode
            const modes = {
                'js': 'javascript', 'jsx': 'javascript', 'ts': 'typescript', 'tsx': 'typescript',
                'html': 'html', 'htm': 'html',
                'css': 'css', 'scss': 'scss', 'less': 'less',
                'json': 'json',
                'xml': 'xml', 'svg': 'xml',
                'sql': 'sql',
                'py': 'python',
                'rb': 'ruby',
                'php': 'php',
                'java': 'java',
                'c': 'c_cpp', 'cpp': 'c_cpp', 'h': 'c_cpp', 'cs': 'csharp',
                'go': 'golang',
                'rs': 'rust',
                'swift': 'swift',
                'kt': 'kotlin',
                'lua': 'lua',
                'sh': 'sh', 'bat': 'batchfile', 'ps1': 'powershell',
                'yml': 'yaml', 'yaml': 'yaml',
                'ini': 'ini', 'toml': 'toml', 'conf': 'ini', 'properties': 'properties',
                'md': 'markdown',
                'txt': 'text', 'log': 'text', 'rtf': 'text',
                'env': 'text'
            };

            const ext = file.name.split('.').pop().toLowerCase();
            let mode = modes[ext] || 'text';
            if (file.name === '.env' || file.name.endsWith('.env')) mode = 'ini';

            this.editor.session.setMode(`ace/mode/${mode}`);
            this.editor.setValue(content, -1);

            // Show Modal
            modal.style.display = 'flex';
            this.editor.focus();

        } catch (error) {
            console.error('Failed to open editor:', error);
            showToast('Failed to open editor: ' + error.message, 'error');
        } finally {
            hideLoading();
        }
    }

    closeEditor() {
        document.getElementById('editor-modal').style.display = 'none';
        this.currentEditingFileId = null;
        if (this.editor) {
            this.editor.setValue('', -1);
        }
    }

    async saveEditorContent() {
        if (!this.currentEditingFileId) return;

        try {
            const content = this.editor.getValue();
            const statusEl = document.getElementById('editor-status');

            statusEl.textContent = 'Saving...';

            await api.updateFileContent(this.currentEditingFileId, content);

            statusEl.textContent = 'Saved successfully at ' + new Date().toLocaleTimeString();
            showToast('File saved successfully', 'success');

            // Clear status after delay
            setTimeout(() => {
                if (statusEl.textContent.startsWith('Saved')) {
                    statusEl.textContent = '';
                }
            }, 3000);

        } catch (error) {
            console.error('Failed to save:', error);
            showToast('Failed to save: ' + error.message, 'error');
            document.getElementById('editor-status').textContent = 'Error saving file';
        }
    }
}

// Initialize
let fileManager;
// Wait for DOM
document.addEventListener('DOMContentLoaded', () => {
    // Only init if not already initialized (in case included script runs early?)
    // Actually the script is at end of body.
    if (!fileManager) {
        fileManager = new FileManager();
    }
});
